package locale.zh_cn;

import java.util.ArrayList;
import java.util.Calendar;

import com.avaya.sce.runtime.Format;
import com.avaya.sce.runtime.ILocaleFormatter;
import com.avaya.sce.runtime.InputFormatException;
import com.avaya.sce.runtime.LocaleResults;

public class FormatLocale implements ILocaleFormatter, IPhraseDefs {
	/*--- Standard Formats ---*/

	/* 
	* Standard integer input data is formatted as: [-]#[,]### 
	* Supports integers up to 9 billion
	* output formatted as spoken Numbers, "One-Thousand Fifty-One" given 1051
	*/
	public static final int FORMAT_NN = 101;
	public static final String FORMAT_NN_STR = "Nn";

	/* 
	* Standard decimal number input data is formatted as: [-]#[,]###.### 
	* output formatted as spoken Numbers, "One-Thousand Fifty-One-point-Zero-Five" given 1051.05
	* Three different inflections are supported.
	* For no decimal separator it always forces "point zero" to be spoken 
	*/
	public static final int FORMAT_NDN = 201;
	public static final String FORMAT_NDN_STR = "NDn";

	/* 
	* Date long input data is formatted as: YYYYMMDD
	* output formatted as "1998 December 1st " given 12011998
	*/
	public static final int FORMAT_DYYMD = 301;
	public static final String FORMAT_DYYMD_STR = "DYYMD";

	/* 
	* Date short input data is formatted as: YYYYMMDD 
	* output formatted as "December 1st" given 12011998
	*/
	public static final int FORMAT_DMD = 302;
	public static final String FORMAT_DMD_STR = "DMD";
	
	/* 
	* Date short input data is formatted as: YYYYMMDD 
	* output formatted as "1998-1911 year 12 month 01 day" given 12011998
	*/
	public static final int FORMAT_DTWTTMD = 303;
	public static final String FORMAT_DTWTTMD_STR = "DTWTTMD";
	
	/* 
	* Time short input data is formatted as: HH:MM:SS
	* output formatted as  , "PM One hour Thirty mints" given 13:30
	* 12 output hour format, seconds optional.
	*/
	public static final int FORMAT_THMAM = 401;
	public static final String FORMAT_THMAM_STR = "THMAM";

	/* 
	* Time short 24 hour input data is formatted as: HH:MM:SS
	* output formatted as "Thirteen Hours Thirty Mints" given 13:30
	* 24 output hour format, seconds optional.
	*/
	public static final int FORMAT_TH24MSSL = 402;
	public static final String FORMAT_TH24MSSL_STR = "TH24MSSL";

	/* 
	* Standard currency number input data is formatted as: [-]#[,]###.### 
	* output formatted as spoken Numbers, "One-Thousand Fifty-One Quai" given 1051.05
	* it ignores numbers after decimal point 
	*/
	public static final int FORMAT_N$ = 501;
	public static final String FORMAT_N$_STR = "N$";

	/* 
	* Standard currency number input data is formatted as: [-]#[,]###.### 
	* output formatted as spoken Numbers, "One-Thousand Fifty-One Quai Five fun" given 1051.05
	* it takes only first 2 digits after decimal point 
	* it is spoken as "Kuai, Mao, Fei".
	*/
	public static final int FORMAT_N$2 = 502;
	public static final String FORMAT_N$2_STR = "N$2";

	/* 
	* Standard currency number input data is formatted as: [-]#[,]###.### 
	* output formatted as spoken Numbers, "One-Thousand-Fifty-One-Point-Zero-Five-Seven-Five kuai" given 1051.0575
	* it speaks the number as general decimal number with "kuai" appended 
	*/
	public static final int FORMAT_N$D = 503;
	public static final String FORMAT_N$D_STR = "N$D";
	
	/* spoken as chinese way of "Yuan, Jiao, Fen" 
	* Standard currency number input data is formatted as: [-]#[,]###.### 
	* output formatted as spoken Numbers, "One-Thousand Fifty-One Yuan" given 1051.05
	* it ignores numbers after decimal point 
	*/
	public static final int FORMAT_NYD0 = 504;
	public static final String FORMAT_NYD0_STR = "NYD0";

	/* spoken as chinese currency in the way of "Yuan, Jiao, Fen" 
	* Standard currency number input data is formatted as: [-]#[,]###.### 
	* output formatted as spoken Numbers, "One-Thousand Fifty-One Yuan Five fun" given 1051.05
	* it takes only first 2 digits after decimal point 
	*/
	public static final int FORMAT_NYD2 = 505;
	public static final String FORMAT_NYD2_STR = "NYD2";

	/* spoken as chinese currency in the way of "Yuan, Jiao, Fen" 
	* Standard currency number input data is formatted as: [-]#[,]###.### 
	* output formatted as spoken Numbers, "One-Thousand-Fifty-One-Point-Zero-Five-Seven-Five Yuan" given 1051.0575
	* it speaks the number as general decimal number with "Yuan" appended 
	*/
	public static final int FORMAT_NYD = 506;
	public static final String FORMAT_NYD_STR = "NYD";
	
	
	/* 
	* Standard string is formatted as "CCCCCCCCCCCCCCCCC" 
	* output formatted as spelled digits and letters
	* only 0-9, a-z, A-Z supported (other characters are ignored)
	*/
	public static final int FORMAT_CRMM = 601;
	public static final String FORMAT_CRMM_STR = "Crmm";
	
	// Those are some auxiliary constants for internal purposes  
	public static final int INTG = 1;
	public static final int DECIM = 2;
	
	public static final int ANYCHARS = 0;
	public static final int DECDIGITS = 1;
	
	public static final int MONTHNUM = 1;
	public static final int MONTHSPOKEN = 2;
	
	public static final int TIME12 = 1;
	public static final int TIME24 = 2;
	public static final int TIMESHORT = 1;
	public static final int TIMELONG = 2;
	public static final int TIMELONGEXT = 3;

	public static final int DOLLAR = 1;
	public static final int YUAN = 2;
	
	public static final int DATE_LONG = 1;
	public static final int DATE_MEDIUM = 2;
	public static final int DATE_SHORT = 3;
	public static final int DATE_NOYEAR = 4;
	
	public static final long TRILLION = (long)1e12;

	public FormatLocale() {
	}	

	private int convertFormat(String value, LocaleResults messages) {
		/*
		 * compares the value passed to it from the formatTextToAudio()
		 * below and maps that string vs. the string variables
		 * of this class above, in order to return a corresponding int
		 */
		//System.out.println("Here is the value passed into convertFormat(): "+ value);
		if (value.equals(FORMAT_NN_STR) == true) {
			return (FORMAT_NN);
		} else if (value.equals(FORMAT_NDN_STR) == true) {
			return (FORMAT_NDN); //up to here, the formats are for the Number format
		} else if (value.equals(FORMAT_DYYMD_STR) == true) {
			return (FORMAT_DYYMD);
		} else if (value.equals(FORMAT_DMD_STR) == true) {
			return (FORMAT_DMD);
		} else if (value.equals(FORMAT_DTWTTMD_STR) == true ){
			return (FORMAT_DTWTTMD);//up to here, the formats are for the Date format
		} else if (value.equals(FORMAT_THMAM_STR) == true) {
			return (FORMAT_THMAM);
		} else if (value.equals(FORMAT_TH24MSSL_STR) == true) {
			return (FORMAT_TH24MSSL); //up to here, the formats are for the Time format
		} else if (value.equals(FORMAT_N$_STR) == true) {
			return (FORMAT_N$);
		} else if (value.equals(FORMAT_N$2_STR) == true) {
			return (FORMAT_N$2);
		} else if (value.equals(FORMAT_N$D_STR) == true) {
			return (FORMAT_N$D);
		} else if (value.equals(FORMAT_NYD0_STR) == true) {
			return (FORMAT_NYD0);
		}  else if (value.equals(FORMAT_NYD2_STR) == true) {
			return (FORMAT_NYD2);
		} else if (value.equals(FORMAT_NYD_STR) == true) {
			return (FORMAT_NYD); //up to here, the formats are for the Currency format
		} else if (value.equals(FORMAT_CRMM_STR) == true) {
			return (FORMAT_CRMM);} //this format is for the Character format
			
  //For non-matching formats, resort to default for each group
			
			if (value.startsWith("ND") == true){
				messages.getMessages().add("convertFormat, warning: format \""+ value.toString()+"\" was not resolved. Using default format \""+ FORMAT_NDN_STR + "\"instead");
				return (FORMAT_NDN);
			}	
			if (value.startsWith("N$") == true){
				messages.getMessages().add("convertFormat, warning: format \""+ value.toString()+"\" was not resolved. Using default format \""+ FORMAT_N$D_STR + "\"instead");
				return (FORMAT_N$D);
			}	
			if (value.startsWith("N") == true){
				messages.getMessages().add("convertFormat, warning: format \""+ value.toString()+"\" was not resolved. Using default format \""+ FORMAT_NN_STR + "\"instead");
				return (FORMAT_NN);
			}
			if (value.startsWith("D") == true){
				messages.getMessages().add("convertFormat, warning: format "+ value.toString()+"\" was not resolved. Using default format \""+ FORMAT_DYYMD_STR + "\"instead");
				return (FORMAT_DYYMD);
			}
			if (value.startsWith("T") == true){
				messages.getMessages().add("convertFormat, warning: format \""+ value.toString()+"\" was not resolved. Using default format \""+ FORMAT_THMAM_STR + "\"instead");
				return (FORMAT_THMAM);
			}
			if (value.startsWith("C") == true){
				messages.getMessages().add("convertFormat, warning: format \""+ value.toString()+"\" was not resolved. Using default format \""+ FORMAT_CRMM_STR + "\"instead");
				return (FORMAT_CRMM);
			} 
			return (-1);
		}
	

	/**
	 * @see com.avaya.sce.runtime.ILocaleFormatter#formatTextToAudio(String, Format)
	 */
    public String[] formatTextToAudio(String value, Format formats, LocaleResults messages) throws InputFormatException {
		 
		String[] result = null;
		ArrayList list = null;
     //Format tag is retrieved from the last pair of values delivered by DD generated code (based on <lang>.xml leaf property 	
		Format.FormatAttribute[] formatAttribs = formats.getFormatAttributes();
		int last = formatAttribs.length - 1;
		if (last < 0) throw new InputFormatException("formatTextToAudio: No format information provided!");
		
		String formatTag = formatAttribs[last].getValue();
	
		int format = convertFormat(formatTag, messages);
	
		if (format < 1) throw new InputFormatException("formatTextToAudio: Unrecognized input format");

		switch (format) {
			case FORMAT_NN :
				list = toNumber(value, INTG, messages);
				break;
			case FORMAT_NDN :
				list = toNumber(value, DECIM, messages);
				break;
			case FORMAT_DYYMD :
			    list = toDate(value, format, DATE_MEDIUM, messages);
			    break;
			case FORMAT_DMD :
			    list = toDate(value, format, DATE_SHORT, messages);
				break;
			case FORMAT_DTWTTMD:
				list = toDate(value, format, DATE_MEDIUM,messages);
				break;				
			case FORMAT_THMAM:
				list = toTime(value, TIME12,TIMESHORT,messages);
				break;
			case FORMAT_TH24MSSL:
			    list = toTime(value, TIME24,TIMELONG, messages);
			    break;			    
			case FORMAT_N$:
				list = toCurrency(value, format, DOLLAR, messages);
				break;
			case FORMAT_N$2:
				list = toCurrency(value, format,DOLLAR,  messages);
				break;
			case FORMAT_N$D:
				list = toCurrency(value, format, DOLLAR, messages);
				break;
			case FORMAT_NYD0:
				list = toCurrency(value, format,YUAN, messages);
				break;
			case FORMAT_NYD2:
				list = toCurrency(value, format, YUAN,  messages);
				break;
			case FORMAT_NYD:
				list = toCurrency(value, format, YUAN,  messages);
				break;				
			case FORMAT_CRMM:
				list = toChars(value,  messages);
				break;
				
			default :
				break;
		}

        if (list != null) {
            if (list.size() > 0) {
            	result = (String[]) list.toArray(new String[1]);
            }
        }
		return (result);
	}
    

	/**
	* Convert the input string into a spoken number bases on the format.
	* It can accept generic number format in form:
	* 			
	* [-]#[,]###.### 
	* 
	* All spaces are filtered out (ignored) and the string is terminated after evetual "." or "," (depends on hundreds separator character)   
	* 
	* @param value  string value to convert to clips
	* @param inflection inflection specifier
	* @param messages  returns warnings and info messages 
	* @exception InputFormatException if the value is invalid
	*/
    
	private ArrayList toNumber(String value, int numberType, LocaleResults messages) throws InputFormatException {
		ArrayList list = new ArrayList();
		int i;
		int length;

		//Debug:
		//messages.getMessages().add("DEBUG: toNumber,  value \""+ value.toString() + "\" ");
		
		length = value.length();
		if (length == 0) {
			throw new InputFormatException("toNumber, Null value in the input string");
		}
		
		// String is first analyzed which hundreds and decimal separators are used.
		int commaSepPos=-1, periodSepPos=-1;
		int commaSepCount=0, periodSepCount=0;
		char thousandsSep='.', decimalSep='.';
		for (i=0; i< value.length(); i++){
			if (value.charAt(i)=='.'){
				periodSepPos=i;
				periodSepCount++;
			}
			if (value.charAt(i)==','){
				commaSepPos=i;
				commaSepCount++;
			}
		}
		// if both comma and period is present within the strings, decimal sep. is the rightmost one
		if ((commaSepPos>=0)&&(periodSepPos>=0)){
			thousandsSep=commaSepPos<periodSepPos ? ',' : '.';
			decimalSep=commaSepPos>periodSepPos ? ',' : '.';
		}
		else{
			if (commaSepPos>=0){
				// If comma is the only separator and is present just once then use it:
				// - either as hundreds (digit grouping) separator - for integer formats
				// - or as decimal separator - for decimal formats
				if (commaSepCount==1){
					thousandsSep=(numberType==INTG) ? ',' : ' ';
					decimalSep=(numberType==INTG) ? '.' : ',';
				}
				else{
					thousandsSep=',';
					decimalSep='.';
				}
			}
			else
				if (periodSepPos>=0){
					// If period is the only separator and is present just once then use it:
					// - either as hundreds (digit grouping) separator - for integer formats
					// - or as decimal separator - for decimal formats
					if (periodSepCount==1){
						thousandsSep=(numberType==INTG) ? '.' : ',';
						decimalSep=(numberType==INTG) ? ',' : '.';
					}
					else{
						thousandsSep='.';
						decimalSep=' ';
					}
				}
		}
			
		// Commas and hundreds spearators are filtered out and string is trimmed after eventual decimal point
		int decPointPos=-1;
		String filteredValue = new String(""); 
		for (i=0; i< value.length(); i++){
			if (value.charAt(i)==decimalSep){
				if (numberType==DECIM) decPointPos=i;
				else messages.getMessages().add("toNumber, expecting integer - value \""+ value.toString() + "\" terminated at decimal separator: \""+ filteredValue.toString() + "\"");
				break;
			}
			
			if ((value.charAt(i)==thousandsSep)||(value.charAt(i)==' '));
			else filteredValue +=value.substring(i,i+1);
		}
		
		// Try first to convert it long integer and check if the number is not equal or bigger than one milliard  
		long longNumber;
		try{
			longNumber = Long.parseLong(filteredValue);
		}
		catch (NumberFormatException e){
			throw new InputFormatException("toNumber, invalid value \""+ value.toString() + "\" - could not be converted to integer");
		}
		if (Math.abs(longNumber)>=TRILLION){
			throw new InputFormatException("toNumber, value is too big, can handle values only less than one hundred billion: [-]1e12-1");
		}
		
		// This is a special case when decimal number is negative and whole part is 0 (e.g. -0.1), it must be handled explicitely
		if ((longNumber==0)&&(filteredValue.startsWith("-"))){
			list.add(STD_MINUS);
			longNumber = longNumber * -1;
		}
		
		if (numberType==DECIM){
			speakNumber(longNumber, list, messages);
			list.add(STD_POINT);
			if (decPointPos<0){
				speakChars("0",DECDIGITS, list, messages);
				messages.getMessages().add("toNumber, warning: input \""+ value.toString()+"\" contains no decimal part, forcing to speak 'point zero'");
			}
			else speakChars(value.substring(decPointPos+1),DECDIGITS, list, messages);
		}
		else speakNumber(longNumber,list, messages);
		return (list);
	}
		


	
	/**
	* Convert the input string into a date based on the format.
	* It can accept generic number format in following format based on ISO 8601:
	* 
	* YYYY[-]MM[-]DD
	* 
	* Separators are optional
	*  
	* @param value  string value to convert to clips
	* @param monthType flag for numeric/spoken month
	* @param dateLength date format specifier
	* @param messages  returns warnings and info messages 
	* @exception InputFormatException if the value is invalid
	*/
    
	private ArrayList toDate(String value, int format, int dateLength, LocaleResults messages) throws InputFormatException {
		ArrayList list = new ArrayList();
		int year;
		
		int month, day;
		int length;
		

		//Debug:
		//messages.getMessages().add("DEBUG: toDate,  value \""+ value.toString() + "\" ");
		
		length = value.length();
		if (length == 0) {
			throw new InputFormatException("toDate, Null value in the input string");
		}
		if ((length < 8)&&(length>10)) {
			throw new InputFormatException("toDate, value must be in format of YYYY[-]MM[-]DD");
		}
		int monthOffset=0, dayOffset=0;
		try{
			// First scans the input string for eventual delimiters
			for (int i=0; i< value.length(); i++){
				if (value.charAt(i)=='-'){
					if (monthOffset>0){
						dayOffset=i+1;
						break;
					}
					else monthOffset=i+1;
				}
			}
			//year=Integer.parseInt(value.substring(0, (monthOffset>0) ? monthOffset-1 : 4));
			year=Integer.parseInt(value.substring(0, (monthOffset>0) ? monthOffset-1 : 4));
			month=(monthOffset>0) ? Integer.parseInt(value.substring(monthOffset,dayOffset-1)) : Integer.parseInt(value.substring(4,6));
			day=(dayOffset>0) ? Integer.parseInt(value.substring(dayOffset)) : Integer.parseInt(value.substring(6,8));
		}
		catch (NumberFormatException e){
			throw new InputFormatException("toDate, invalid value for date \""+ value.toString() + "\" - could not be converted to integer");
		}
		if ((day<1)||(day>31)) throw new InputFormatException("toDate, invalid day value \""+ Integer.toString(day) + "\" - must be between 1-31");
		if ((month<1)||(month>12)) throw new InputFormatException("toDate, invalid month value \""+ Integer.toString(month) + "\" - must be between 1-12");

		Calendar calendar=Calendar.getInstance();
		calendar.set(year, month - 1, day);
		
		int dayOfMonth = calendar.get(Calendar.DAY_OF_MONTH);
		if (dayOfMonth != day) {
			throw new InputFormatException("toDate, invalid date value");
		}
		if (format == FORMAT_DYYMD)
           {
           	speakChars((value.substring(0,4)),DECDIGITS, list, messages);
			list.add(STD_YEAR);
			list.add(STD_SIL_DOT_200);
			speakNumber(month,list,messages);
			list.add(STD_MONTH);
			list.add(STD_SIL_DOT_200);
			speakNumber(day,list,messages);
			list.add(STD_DAY);
            }
		else if (format == FORMAT_DMD){
			speakNumber(month,list,messages);
			list.add(STD_MONTH);
			speakNumber(day,list,messages);
			list.add(STD_DAY);
				}
		else{
			list.add(STD_REPUBLIC);
			if (year<1911) {
				throw new InputFormatException("toDate, invalid year value");
			}
			speakNumber(year-1911, list,messages);
			list.add(STD_YEAR);
			speakNumber(month,list,messages);
			list.add(STD_MONTH);
			speakNumber(day,list,messages);
			list.add(STD_DAY);
			}
		return list;
	}
	
	
	/** 
	* Convert the input string into a spoken number bases on the format.
	* It can accept generic number format in form:
	* 			
	* HH[:]MM[:][SS] 
	* 
	* Separators and seconds are optional, must meet common ranges 0-23 for hours, 0-59 for minutes, 0-59 for seconds   
	* 
	* @param value  string value to convert to clips
	* @param timeSystem flag for 12/24h clock system
	* @param timeLength flag for seconds to be spoken or not
	* @param messages  returns warnings and info messages 
	* @exception InputFormatException if the value is invalid
	*/
    
	private ArrayList toTime(String value, int timeSystem, int timeLength, LocaleResults messages) throws InputFormatException {
		ArrayList list = new ArrayList();
		int hours, minutes,seconds;
		int length;

		//Debug:
		//messages.getMessages().add("DEBUG: toTime,  value \""+ value.toString() + "\" ");
		
		length = value.length();
		if (length == 0) {
			throw new InputFormatException("toTime, Null value in the input string");
		}
		if ((length < 4)||(length>8)) {
			throw new InputFormatException("toTime, value must be in format of HH[:]MM[:][SS]");
		}

		int minutesOffset=0, secondsOffset=0;
		try{
			// First scans the input string for eventual delimiters
			for (int i=0; i< value.length(); i++){
				if (value.charAt(i)==':'){
					if (minutesOffset>0){
						secondsOffset=i+1;
						break;
					}
					else minutesOffset=i+1;
				}
			}

			hours=Integer.parseInt(value.substring(0,(minutesOffset>0) ? minutesOffset-1 : 2));
			minutes=(minutesOffset>0) ? 
					(secondsOffset>0) ?  Integer.parseInt(value.substring(minutesOffset, secondsOffset-1)) : Integer.parseInt(value.substring(minutesOffset)) 
					: Integer.parseInt(value.substring(2,4));
			seconds=(secondsOffset>0) ? Integer.parseInt(value.substring(secondsOffset)) : (minutesOffset==0)&&(length==6) ? Integer.parseInt(value.substring(4,6)) : 0;
		}
		catch (NumberFormatException e){
			throw new InputFormatException("toTime, invalid value for time \""+ value.toString() + "\" - could not be converted to integer");
		}
		if (minutesOffset==0 && secondsOffset==0 && length!=6 && length!=4){
			throw new InputFormatException("toDate, value must be either 4 or 6 digit long in format of HHMM[SS] if you do not use separators");
		}

		
		if ((hours<0)||(hours>24)) throw new InputFormatException("toTime, invalid hours value \""+ Integer.toString(hours) + "\" - must be between 0-24");
		if ((minutes<0)||(minutes>59)) throw new InputFormatException("toTime, invalid minutes value \""+ Integer.toString(minutes) + "\" - must be between 0-59");
		if ((seconds<0)||(seconds>59)) throw new InputFormatException("toTime, invalid seconds value \""+ Integer.toString(seconds) + "\" - must be between 0-59");

		// 24 hour system
			if (timeSystem==TIME24){
			if (hours!=2)
			speakNumber(hours,list,messages);
			else  list.add(STD_2_STAR_2);
			if (timeLength==TIMELONG){
				list.add(STD_POINT);
				if (minutes<10){					
					if (minutes==0)	list.add(STD_0);
					else {list.add(STD_0);
					speakNumber(minutes,list,messages);}					
				}
				else speakNumber(minutes,list,messages);
				list.add(STD_MINUTE);				
				}
			}
			else{
				if (hours<5) list.add(STD_LING_SHUN);
			  			
		    	else if ((hours >=5) && (hours < 8)) list.add(STD_ZHAO_SHANG);
		    		
			    else if (hours < 12)  list.add(STD_SHANG_WOO);	
			    					
			    else if (hours < 13)  list.add(STD_TUNG_WOO);	
			    					
			    else if (hours < 18)  list.add(STD_SIYA_WOO);
			    					    	
			    else {list.add(STD_WAN_SHANG);}	
				
				if (hours>12) {
					if (hours==14) list.add(STD_2_STAR_2); 
					else speakNumber(hours-12, list, messages);
				}
				else if(hours<13)	{
			            if (hours==2) 	list.add(STD_2_STAR_2);
			            else speakNumber(hours ,list,messages);}
				list.add(STD_POINT); 				
						    		
			    if (minutes<10){
			    		if (minutes==0)	list.add(STD_0);
			    		else {
			    			list.add(STD_0);
			    			speakNumber(minutes,list,messages);}
			    		}
			    else speakNumber(minutes,list,messages);
			    	list.add(STD_MINUTE);		 	    	
			  	}			
		return (list);
	}

 
	/**
	* Convert the input string into a spoken number with currency symbol bases on the format.
	* It can accept generic number format in form:
	* 			
	* [-]#[,]###.### 
	* 
	* All spaces, digit grouping spearators and dollar signs are filtered out (ignored)    
	* 
	* @param value  string value to convert to clips
	* @param format format specifier
	* @param messages  returns warnings and info messages 
	* @exception InputFormatException if the value is invalid
	*/
    
	private ArrayList toCurrency(String value, int format, int currencyType,LocaleResults messages) throws InputFormatException {
		ArrayList list = new ArrayList();
		int i;
		int length;

		//Debug:
		//messages.getMessages().add("DEBUG: toCurrency,  value \""+ value.toString() + "\" ");
		
		length = value.length();
		if (length == 0) {
			throw new InputFormatException("toCurrency, Null value in the input string");
		}
		
		// String is first analyzed which thousands and decimal separators are used.
		int commaSepPos=-1, periodSepPos=-1;
		int commaSepCount=0, periodSepCount=0;
		char thousandsSep='.', decimalSep='.';
		for (i=0; i< value.length(); i++){
			if (value.charAt(i)=='.'){
				periodSepPos=i;
				periodSepCount++;
			}
			if (value.charAt(i)==','){
				commaSepPos=i;
				commaSepCount++;
			}
		}
		// if both comma and period is present within the strings, decimal sep. is the rightmost one
		if ((commaSepPos>=0)&&(periodSepPos>=0)){
			thousandsSep=commaSepPos<periodSepPos ? ',' : '.';
			decimalSep=commaSepPos>periodSepPos ? ',' : '.';
		}
		else{
			if (commaSepPos>=0){
				// If comma is the only separator and is present just once then use it:
				// as decimal separator (treat it always as decimal number)
				if (commaSepCount==1){
					thousandsSep=' ';
					decimalSep=',';
				}
				else{
					thousandsSep=',';
					decimalSep='.';
				}
			}
			else
				if (periodSepPos>=0){
					// If period is the only separator and is present just once then use it:
					// as decimal separator (treat it always as decimal number)
					if (periodSepCount==1){
						thousandsSep=',';
						decimalSep='.';
					}
					else{
						thousandsSep='.';
						decimalSep=',';
					}
				}
		}

		
		// Spaces, digit grouping (thousands separators and dollar signs are filtered out and string is trimmed after eventual decimal point
		String filteredValue = new String(""); 
		int decPointPos=-1;
		for (i=0; i< value.length(); i++){
			if (value.charAt(i)==decimalSep){
				decPointPos=i;
				break;
			}
			if ((value.charAt(i)==' ')||(value.charAt(i)=='$')||(value.charAt(i)==thousandsSep));
			else filteredValue +=value.substring(i,i+1);
		}
		
		// Try first to convert it long integer and check if the number is not equal or bigger than one milliard  
		long longNumber;
		try{
			longNumber = Long.parseLong(filteredValue);
		}
		catch (NumberFormatException e){
			throw new InputFormatException("toCurrency, invalid value \""+ value.toString() + "\" - could not be converted to integer");
		}
		if (Math.abs(longNumber)>= TRILLION){
			throw new InputFormatException("toCurrency, value too big, can handle values only less than quadrillion: [-]1e15-1");
		}
		
		if (currencyType == DOLLAR){
			list.add(STD_MEIJIN);
			// Negative value must be handled explicitely
			if (filteredValue.startsWith("-")){
				list.add(STD_MINUS);
				longNumber = longNumber * -1;
			}
			if (format==FORMAT_N$){
				
				speakNumber(longNumber, list, messages);
				list.add(STD_QUAI);}
			if (format==FORMAT_N$2){
				
				speakNumber(longNumber,list, messages);
				list.add(STD_QUAI);
				int cents=0;
				if (decPointPos<0){
					messages.getMessages().add("toCurrency, warning: input \""+ value.toString()+"\" contains no decimal part, forcing to speak 'zero cents'");
				}
				else{
					try{
						cents=Integer.parseInt(value.substring(decPointPos+1, decPointPos+1+Math.min(value.substring(decPointPos+1).length(),2)));
						if (value.substring(decPointPos+1).length()<2) cents*=10;
					}
					catch(NumberFormatException e){
						throw new InputFormatException("toCurrency, invalid value \""+ value.toString() + "\" - decimal part could not be converted to integer");
					}	
				}
				
				speakNumber(((int)cents/10),list, messages);
				list.add(STD_MAO);
				speakNumber(cents%10,list, messages);
				list.add(STD_FUN);
				
				}
			else if (format==FORMAT_N$D){
			
				speakNumber(longNumber,list, messages);
				list.add(STD_POINT);
				if (decPointPos<0){
					speakChars("0",DECDIGITS, list, messages);
					messages.getMessages().add("toCurrency, warning: input \""+ value.toString()+"\" contains no decimal part, forcing to speak 'point zero'");
				}
				else speakChars(value.substring(decPointPos+1),DECDIGITS, list, messages);
				list.add(STD_QUAI);
			}
		}
		else {
			// Negative value must be handled explicitely
			if (filteredValue.startsWith("-")){
				list.add(STD_MINUS);
				longNumber = longNumber * -1;
			}

			if (format==FORMAT_NYD0){
				speakNumber(longNumber, list, messages);
				list.add(STD_YUAN);
			}
		
			else if(format==FORMAT_NYD2){
			speakNumber(longNumber,list, messages);
			list.add(STD_YUAN);
			
			int cents=0;
			if (decPointPos<0){
				messages.getMessages().add("toCurrency, warning: input \""+ value.toString()+"\" contains no decimal part, forcing to speak 'zero cents'");
			}
			else{
				try{
					cents=Integer.parseInt(value.substring(decPointPos+1, decPointPos+1+Math.min(value.substring(decPointPos+1).length(),2)));
					if (value.substring(decPointPos+1).length()<2) cents*=10;
				}
				catch(NumberFormatException e){
					throw new InputFormatException("toCurrency, invalid value \""+ value.toString() + "\" - decimal part could not be converted to integer");
				}	
			}
			
			speakNumber(((int)cents/10),list, messages);
			list.add(STD_CHIAO);
			speakNumber(cents%10,list, messages);
			list.add(STD_FUN);
		
		}		
		else {
			speakNumber(longNumber,list, messages);
			list.add(STD_POINT_STAR_2);
			if (decPointPos<0){
				speakChars("0",DECDIGITS, list, messages);
				messages.getMessages().add("toCurrency, warning: input \""+ value.toString()+"\" contains no decimal part, forcing to speak 'point zero'");
			}
			else speakChars(value.substring(decPointPos+1),DECDIGITS, list, messages);
			list.add(STD_YUAN);
			}
		}
		
		return (list);
	}

	/**
	* Convert the input string of characters into individual characters.
	* It can accept letters a-z (case insensitive) and digits 0-9, others will be ignored
	*  
	* @param value  string value to convert to clips
	* @param inflection inflection specifier
	* @exception InputFormatException if the value is invalid
	*/
    		
	private ArrayList toChars(String value, LocaleResults messages) throws InputFormatException {
		ArrayList list = new ArrayList();
		int length = value.length();
		if (length == 0) {
			throw new InputFormatException("toChars, empty string");
		}
		speakChars(value, ANYCHARS,  list, messages);
		return list;
	}
	
	
	/**
	* Auxiliary method:
	* Convert double (integer) value into a spoken number based on the inflection.
	* 
	* @param number  value to convert to clips per the format specifier.
	* @param inflection inflection specifier
	* @param list  buffer for phrases to be spoken  
	* @param messages  array of hint messages 
	*/
    
	private void speakNumber(double number, ArrayList list, LocaleResults messages){
		// Speaking minus for negative value
		if (number < 0){
			list.add(STD_MINUS);
			number = number * -1;
		}
		
		// speak number of one hundred millions to trillions  
		if (number>=1e8){
			int hundredmillions=(int)(number/1e8);
			if (hundredmillions==1) list.add(STD_1_STAR_4);
			else if (hundredmillions==2) list.add(STD_2_STAR_3);
			else speakNumber(hundredmillions, list, messages);
			list.add(STD_100000000);
			if ((number%1e8)>0) {
				if ((number%1e8)<10000000) list.add(STD_0);
				if (((number%1e8)>9)&&((number%1e8)<20)) list.add(STD_1); // this is a fix to wi00231368 
				speakNumber((int)(number%1e8),list, messages);
			} 
		}
		else {
			if (number>=10000)
			// speak number of ten thousand  
			{
				int tenthousand=(int)(number/10000);
				if (tenthousand==1) list.add(STD_1_STAR_4);
				else if (tenthousand==2) list.add(STD_2_STAR_3);
				else speakNumber(tenthousand, list, messages);
				list.add(STD_10000);
				if ((number%10000)>0) {
					if ((number%10000)<1000) list.add(STD_0);
					if (((number%10000)>9)&&((number%10000)<20)) list.add(STD_1); // this is a fix to wi00231368 
					speakNumber((int)(number%10000),list, messages);}
			}
			else {
				if (number>=1000){
				// speak number of thousands  
					int thousand=(int)(number/1000);
					if (thousand==1) list.add(STD_1_STAR_4);
					else if (thousand==2) list.add(STD_2_STAR_3);
					else speakNumber(thousand, list, messages);
					list.add(STD_1000);
					if ((number%1000)>0) {
						if ((number%1000)<100) list.add(STD_0);
						if (((number%1000)>9)&&((number%1000)<20)) list.add(STD_1); // this is a fix to wi00231368 
						speakNumber((int)(number%1000),list, messages);}
				}
				else {
					if (number>=100){
					// speak number of hundreds  
						int hundred=(int)(number/100);
						if (hundred==2) list.add(STD_2_STAR_2);
						else if (hundred==5) list.add(STD_5_STAR_2);
						else if (hundred==9) list.add(STD_9_STAR_2);
						else if (hundred==1) list.add(STD_1_STAR_4);
						else speakNumber(hundred,list, messages);
						String bai =STD_100;
						int tens=(int)(number%100);
						if (((tens>49)&&(tens<60))||(tens>89)) bai=STD_100_STAR_2;
						
						list.add(bai);
						if (tens>0) {
							if (tens<10) list.add(STD_0);
							if ((tens>9)&&(tens<20)) list.add(STD_1);  // this is a fix to wi00231368 
							speakNumber(tens, list, messages);};
					}
					else {
						if (number>=20)
						// speak number of tens  
						{
							speakNumber(number/10,list, messages);
							String ten = STD_10;
							
							list.add(ten);
							if ((number%10)>0)	{speakNumber((int)(number%10), list, messages);}
						}
						else {
							if (number>=10)
							{	
								list.add(STD_10);
								if ((number%10)>0)	{speakNumber((int)(number%10), list, messages);}
							}
						
							else {
								if(number >=0){
						
							// speak units
							
								String units=NumberUpTo10Phrases[(int)number];
								list.add(units);
								}
							}
						}
					}
				}
			}
		}
	}

	/**
	* Auxiliary method:
	* Convert string of characters in a row of characters, last character spoken based on the inflection.
	* 
	* @param string  string value to convert to audio clips per other parameters specified.
	* @param inflection inflection specifier
	* @param stringType  identifies if string should be spoken entirely or trimmed after first non-digit letter 
	* @param list  buffer for phrases to be spoken  
	* @param messages  array of hint messages 
	*/
    
	private void speakChars(String string, int stringType, ArrayList list, LocaleResults messages){
		int i;
		// Input string is first filtered out for invalid (unsupported) characters or truncated before first on non-digit character (for DECDIGITS) 
		String filteredString=new String("");
		for (i=0; i< string.length(); i++){
			if ((stringType==DECDIGITS)&&(Character.isDigit(string.charAt(i))==false)){
				messages.getMessages().add("speakChars, non-digit letter encountered, string trimmed to \""+ filteredString.toString() + "\"");
				break;
			}
			if (Character.isLetterOrDigit(string.charAt(i))==false)
				messages.getMessages().add("speakChars, unsupported character \""+ string.charAt(i) + "\" skipped");
			else filteredString +=string.substring(i,i+1);
		}
		for (i=0; i< filteredString.length(); i++){
			String charPhrase;
			if (Character.isDigit(filteredString.charAt(i))==true) charPhrase=NumberUpTo10Phrases [filteredString.charAt(i)-'0'];
			else charPhrase=STD_LETTERS[Character.toUpperCase(filteredString.charAt(i))-'A'];	

			list.add(charPhrase);
		}	
	}

}
	
        




